/*
 * Copyright 2024 S. V. Nickolas.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the Software), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF, OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

#include <bios.h>
#include <dos.h>
#include <stdio.h>

#include "cua.h"
#include "types.h"

#define FILE_IS_NFDISK_C
#include "nfdisk.h"
#undef FILE_IS_NFDISK_C

#define MAXHD 8

static char *partab[]={"\x01" "DOS (Small)  ",
                       "\x04" "DOS (Medium) ",
                       "\x05" "Extended     ",
                       "\x06" "DOS (Large)  ",
                       "\x07" "OS/2 or NTFS ",
                       "\x0B" "DOS (Jumbo)  ",
                       "\x0C" "DOS (Jumbo)  ",
                       "\x0E" "DOS (Large)  ",
                       "\x11" "OS/2 Boot Mgr",
                       "\x12" "Recovery/BIOS",
                       "\x27" "Recovery/BIOS",
                       "\x36" "OS/2 (JFS)   ",
                       "\x00" "Non-DOS      "};

struct dimension
{
 uint8_t bios_id;
 uint16_t cyls;
 uint8_t secs;
 uint8_t heads;
} hd[MAXHD];

struct partition
{
 uint8_t status;
 uint8_t s_head;
 uint16_t s_chs_cx; /* Use cx2cyls() and cx2secs() to break this down
                       or makecx() to encode it from cyls and secs */
 uint8_t type;
 uint8_t e_head;
 uint16_t e_chs_cx;
 uint32_t lbastart;
 uint32_t lbalen;
};

struct mbr
{
 uint8_t ipl[446];
 struct partition part[4];
 uint16_t signature;
} mbr;

uint8_t hds;

static union REGS regs;

struct diskinfo_t di;

uint16_t makecx (uint16_t cyls, uint8_t secs)
{
 uint16_t x;
 
 x=((cyls&0xFF)<<8)|((cyls&0x0300)>>2);
 x|=(secs&0x3F);
 return x;
}

uint16_t cx2cyls (uint16_t cx)
{
 uint16_t x;
 
 x=(cx&0xFF00)>>8;
 x|=((cx&0x00C0)<<2);
 return x;
}

uint8_t cx2secs (uint16_t cx)
{
 return cx&0x003F;
}

int rdsec (void far *where, uint8_t drive, uint16_t c, uint8_t h, uint8_t s)
{
 uint16_t r;
 
 di.buffer=where;
 di.drive=drive;
 di.track=c;
 di.head=h;
 di.sector=s;
 di.nsectors=1;
 
 r=_bios_disk(2, &di);
 if (!(r&0xFF00)) return 0;
 return ((r&0xFF)!=0x11); /* it quacks like a duck, but it's a LOON! */
}

char *getfmt (uint8_t type)
{
 uint8_t t;
 
 if (!type) return "(Unallocated)";
 for (t=1; t; t++)
 {
  if (!*partab[t]) return partab[t]+1;
  if (*(partab[t])==type) return partab[t]+1;
 }
 
 /* You know what they say about things that Should Never Happen... */
 return "InternalError";
}

int main (int argc, char **argv)
{
 char *p;
 uint16_t t1, t2;
 
 initty();
 
 /* 
  * Find out how many hard drives are installed.
  * Die screaming if no hard drives exist.
  * Otherwise, enumerate them.
  */
 hds=*(uint8_t *)(MK_FP(0x0040, 0x0075));
 if (!hds)
 {
  msgbox(M_NOHD, M_ALERT);
  deinitty();
  return 1;
 }
 if (hds>MAXHD) hds=MAXHD; /* limiter */
 for (t1=0, t2=0x0080; (t1<hds)&&(t2<0x0100); t2++)
 {
  regs.h.ah=0x08;
  regs.h.dl=t2;
  int86(0x13, &regs, &regs);
  if (regs.x.cflag)
   continue; /* failed */

  hd[t1].bios_id=t2;
  hd[t1].cyls=cx2cyls(regs.x.cx);
  hd[t1].secs=cx2secs(regs.x.cx);
  hd[t1].heads=regs.h.dh;
  t1++;
 }
 if (t1<hds) hds=t1;
 
 deinitty();
 printf ("debug: Enumerated %u drives:\n", hds);
 for (t1=0; t1<hds; t1++)
 {
  printf ("debug:   Drive %u: ID %02X  C=%u, H=%u, S=%u\n", 
          t1, hd[t1].bios_id, hd[t1].cyls, hd[t1].heads, hd[t1].secs);
  if (rdsec(&mbr, hd[t1].bios_id, 0, 0, 1))
  {
   printf ("debug:   (Could not read MBR.)\n");
  }
  else
  {
   uint16_t c1, s1, c2, s2;
   
   if (mbr.signature==0xAA55)
   {
    for (t2=0; t2<4; t2++)
    {
     c1=cx2cyls(mbr.part[t2].s_chs_cx);
     s1=cx2secs(mbr.part[t2].s_chs_cx);
     c2=cx2cyls(mbr.part[t2].e_chs_cx);
     s2=cx2secs(mbr.part[t2].e_chs_cx);
     printf ("debug:     %u. S=%5u,%3u,%2u E=%5u,%3u,%2u  [%s]\n",
             t2, c1, mbr.part[t2].s_head, s1, c2, mbr.part[t2].e_head, s2,
             getfmt(mbr.part[t2].type));
    }
   }
   else
   {
    printf ("debug:   (MBR signature not found.)\n");
   }
  }
 }
 printf ("debug: Enumeration complete.\n");
 return 0;
}
