/*
 * Copyright 2024 S. V. Nickolas.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the Software), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF, OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

/*
 * This is an almost-CUA front end for NFDISK.
 * 
 * The following should be considered:
 * 
 *   1. There is no option at this time for online help.
 *      But the user interface is a little more intuitive than FDISK.
 *   2. The input dialogs (inputbox_int and inputbox_vol) are not full-fledged
 *      text entry boxes.  They're just enough to do the job they need to do
 *      in the specific FDISK context and no more.
 *   3. There is only enough tooling for the subset of CUA required by NFDISK.
 * 
 * It was coded defensively with Microsoft C 5.1 in mind but was only tested
 * with Watcom C.
 */

#include <bios.h>
#include <dos.h>
#include <stdio.h>
#include <string.h>

#include "cua.h"
#include "types.h"

#define FILE_IS_CUA_C
#include "nfdisk.h"
#undef  FILE_IS_CUA_C

static uint8_t columns, lines;
static uint8_t mda;

int must_reset = 0;

static uint16_t csrstat;

static union REGS regs;

/* Execute a far jump to F/FFF0, the 8088 reset vector. */
#ifdef __WATCOMC__ /* MSC is too stupid for this */
static void (far *reset8088)(void)=MK_FP(0xFFFF, 0x0000);
#endif

/* Place a character with given attributes on the screen. */
void putchattr (uint8_t c, uint8_t a, uint8_t x, uint8_t y)
{
 uint8_t far *p;
 uint16_t addr;
 
 if ((x>columns)||(y>lines)) return;
 
 addr=((y*columns)+x);
 addr<<=1;
 p=MK_FP((mda?0xB000:0xB800),addr);
 p[0]=c; p[1]=a;
}

/* Write a string with given attributes on the screen. */
void putsattr (char *s, uint8_t a, uint8_t x, uint8_t y)
{
 while (*s)
  putchattr(*(s++), a, x++, y);
}

/*
 * Draw a box with given attributes on the screen.
 * Add a frame with the appropriate box-drawing characters.
 */
void box (uint8_t a, uint8_t x1, uint8_t y1, uint8_t x2, uint8_t y2)
{
 uint8_t x, y;
 
 /* Corners */
 putchattr(218, a, x1, y1);
 putchattr(191, a, x2, y1);
 putchattr(192, a, x1, y2);
 putchattr(217, a, x2, y2);
 
 /* Horizontal lines */
 for (x=x1+1; x<x2; x++)
 {
  putchattr(196, a, x, y1);
  putchattr(196, a, x, y2);
 }
 
 /* Vertical lines and fill */
 for (y=y1+1; y<y2; y++)
 {
  putchattr(179, a, x1, y);
  putchattr(179, a, x2, y);
  for (x=x1+1; x<x2; x++)
   putchattr(' ', a, x, y);
 }
}

/*
 * Draw a scrollbar (for decorative purposes only).
 * Since NFDISK will only use these for the partition list there is no need
 * for horizontal scrollbars, so this function only draws vertical bars.
 */
void scroll (uint8_t start, uint8_t end,
             uint8_t x, uint32_t max, uint32_t pos)
{
 uint8_t barlen, barpos, bary, t;
 
 /* Sanity check */
 if (end<start) return;
 
 /* Figure out where to place the grabber */
 barlen=(end-start)-2;
 barpos=(pos*barlen)/max;
 bary=(start+1)+barpos;
 
 /* Arrowheads */
 putchattr(24, 0x70, x, start);
 putchattr(25, 0x70, x, end);
 
 /* Draw the rest */
 for (t=(start+1); t<end; t++)
  putchattr ((t==bary)?219:178, 0x07, x, t);
}

/* Width needed to represent a string. */
static uint8_t linwid (char *s)
{
 char *p, *q;
 
 uint8_t accum;
 
 p=s;
 accum=0;
 while (p)
 {
  if (NULL!=(q=strchr(p, '\n')))
  {
   if ((q-p)>accum) accum=(q-p);
  }
  else
  {
   if (strlen(p)>accum) accum=strlen(p);
  }
  p=q;
  if (p) p++;
 }
 
 return accum;
}

/* Height needed to represent a string. */
static uint8_t linhgt (char *s)
{
 char *p;
 
 uint8_t accum;
 
 p=s;
 accum=0;
 while (p)
 {
  accum++;
  p=strchr(p, '\n');
  if (p) p++;
 }
 
 return accum;
}

/* Copy a string, stopping at '\n' if found. */
static char *strcpy2eol (char *t, char *s)
{
 while (1)
 {
  if (!(*s))
  {
   *t=0;
   return NULL;
  }
  
  if (*s=='\n')
  {
   *t=0;
   return s+1;
  }
  
  *(t++)=*(s++);
 }
}

/* Draw a dialog box with an <OK> button. */
void msgbox (char *s, char *title)
{
 uint8_t tstr[256];
 char *p;
 
 uint16_t h, w, l, m, x1, x2, y1, y2, c;
 
 h=linhgt(s)+4;
 w=linwid(s)+4;
 
 if (linwid(s)<strlen(M_BTN_OK)) w=strlen(M_BTN_OK)+4;

 x1=(columns>>1)-(w>>1);
 x2=(columns>>1)+(w>>1);
 y1=(lines>>1)-(h>>1);
 y2=(lines>>1)+(h>>1);
 
 box(0x70, x1, y1, x2, y2);
 c=(columns>>1)-(strlen(title)>>1);
 putsattr(title, 0x70, c, y1);
 putchattr(' ', 0x70, c-1, y1);
 putchattr(' ', 0x70, c+strlen(title), y1);
 
 p=s;
 l=(lines>>1)-(linhgt(s)>>1);
 m=(columns>>1)-(linwid(s)>>1);
 while (p)
 {
  p=strcpy2eol(tstr, p);
  putsattr(tstr, 0x70, m, l++);
 }
 
 c=(columns>>1)-(strlen(M_BTN_OK)>>1);
 putsattr(M_BTN_OK, 0x70, c, y2);
 _bios_keybrd(0);
}

/*
 * Request a numeric input, 0-65535.
 * Returns -1 if ESC pressed (this is why a signed long is needed when the
 * input is an unsigned short).
 */
int32_t inputbox_int (char *s, char *title, uint16_t initial)
{
 uint8_t tstr[256];
 char *p;
 
 uint16_t h, w, l, m, x1, x2, y1, y2, c, n;
 uint8_t lokcan, xok, xcan;
 
 n=initial;
 
 h=linhgt(s)+4;
 w=linwid(s)+4;

 lokcan=strlen(M_BTN_OK)+strlen(M_BTN_CANCEL)+2;
 
 if (linwid(s)<lokcan) w=lokcan+4;

 x1=(columns>>1)-(w>>1);
 x2=(columns>>1)+(w>>1);
 y1=(lines>>1)-(h>>1);
 y2=(lines>>1)+(h>>1);
 y1--;
 y2++;
 
 box(0x70, x1, y1, x2, y2);
 c=(columns>>1)-(strlen(title)>>1);
 putsattr(title, 0x70, c, y1);
 putchattr(' ', 0x70, c-1, y1);
 putchattr(' ', 0x70, c+strlen(title), y1);
 
 p=s;
 l=(lines>>1)-(linhgt(s)>>1);
 m=(columns>>1)-(linwid(s)>>1);
 while (p)
 {
  p=strcpy2eol(tstr, p);
  putsattr(tstr, 0x70, m, l++);
 }
 
 xok=(columns>>1)-(lokcan>>1);
 xcan=xok+2+strlen(M_BTN_OK);
 putsattr(M_BTN_OK, 0x70, xok, y2);
 putsattr(M_BTN_CANCEL, 0x70, xcan, y2);

 putsattr("[      ]", 0x70, (columns>>1)-3, l+1);
 sprintf(tstr, "%u", n);
 
 while (1)
 {
  uint16_t c;
  
  putsattr("      ", 0x70, (columns>>1)-2, l+1);
  putsattr(tstr, 0x70, (columns>>1)-2+(5-strlen(tstr)), l+1);
  putchattr(' ', 0x07, (columns>>1)+3, l+1);
  c=_bios_keybrd(0);
  c&=0xFF; /* only matters if ASCII */
  if (c==27) return -1;
  if (c==13) return n;
  if (c==8)
  {
   n/=10;
   sprintf(tstr, "%u", n);
   continue;
  }
  if ((c>='0')&&(c<='9'))
  {
   if (n>6553) continue; /* would produce invalid result */
   n*=10;
   n+=(c-'0');
   sprintf(tstr, "%u", n);
   continue;
  }
 }
 
 return n;
}

char *inputbox_vol (char *s, char *title)
{
 uint8_t tstr[256];
 char *p;
 
 uint16_t h, w, l, m, x1, x2, y1, y2, c, n;
 uint8_t lokcan, xok, xcan;
 
 h=linhgt(s)+4;
 w=linwid(s)+4;

 lokcan=strlen(M_BTN_OK)+strlen(M_BTN_CANCEL)+2;
 
 if (linwid(s)<lokcan) w=lokcan+4;

 x1=(columns>>1)-(w>>1);
 x2=(columns>>1)+(w>>1);
 y1=(lines>>1)-(h>>1);
 y2=(lines>>1)+(h>>1);
 y1--;
 y2++;
 
 box(0x70, x1, y1, x2, y2);
 c=(columns>>1)-(strlen(title)>>1);
 putsattr(title, 0x70, c, y1);
 putchattr(' ', 0x70, c-1, y1);
 putchattr(' ', 0x70, c+strlen(title), y1);
 
 p=s;
 l=(lines>>1)-(linhgt(s)>>1);
 m=(columns>>1)-(linwid(s)>>1);
 while (p)
 {
  p=strcpy2eol(tstr, p);
  putsattr(tstr, 0x70, m, l++);
 }
 
 xok=(columns>>1)-(lokcan>>1);
 xcan=xok+2+strlen(M_BTN_OK);
 putsattr(M_BTN_OK, 0x70, xok, y2);
 putsattr(M_BTN_CANCEL, 0x70, xcan, y2);
 
 memset(tstr, 0, 12);
 putsattr("[            ]", 0x70, (columns>>1)-6, l+1);
 
 while (1)
 {
  uint16_t c;
  
  putsattr("            ", 0x70, (columns>>1)-5, l+1);
  putsattr(tstr, 0x70, (columns>>1)-5, l+1);
  putchattr(' ', 0x07, (columns>>1)-5+strlen(tstr), l+1);
  c=_bios_keybrd(0);
  c&=0xFF; /* only matters if ASCII */
  if (c==27) return NULL;
  if (c==13) return tstr;
  if (c==8)
  {
   if (*tstr)
    tstr[strlen(tstr)-1]=0;
   continue;
  }
  if ((c>='a')&&(c<='z')) c-=32;
  /*
   * Filter invalid characters.
   * Although space is part of the official "nope" list, it has always been
   * allowed in volume labels.
   */
  if ((c<32)||(c==34)||((c>=42)&&(c<45))||((c>45)&&(c<48))||
      ((c>57)&&(c<64))||(c==124)||(c>126))
  {
   continue;
  }
  /* Out of space. */
  if (strlen(tstr)==11)
  {
   continue;
  }
  tstr[strlen(tstr)+1]=0;
  tstr[strlen(tstr)]=c;
 }
}

/*
 * Pops up a message box alerting the user that we are about to reboot.
 * After waiting for a keypress, thrashes the cache and immediately reboots.
 */
void reboot (void)
{
#ifndef __WATCOMC__
 void (far *reset8088)(void);
#endif

 msgbox(M_REBOOT, M_ALERT);
 regs.h.ah=0x0D; /* RESET */
 int86(0x21, &regs, &regs);
 *((uint16_t far *)MK_FP(0x0040, 0x0072))=0x1234;
 
#ifdef __WATCOMC__
 reset8088();
#else
 reset8088=MK_FP(0xFFFF, 0x0000);
 reset8088();
#endif
}

void initty (void)
{
 uint8_t x, y;
 
 int isega;
 
 isega=0;
 
 /*
  * Identify video mode and screen dimensions.
  * 
  * If video mode is 7, then screen buffer is at B/0000 and the display can be
  * assumed to follow MDA rules.  If 2 or 3, then screen buffer is at B/8000
  * and the display can be assumed to follow CGA rules.  If 0 (BW40), set it
  * to 2 (BW80); otherwise set it to 3 (CO80).
  */
 lines=25;
 regs.h.ah=0x12;
 regs.x.bx=0xFF10;
 regs.x.cx=0xFFFF;
 int86(0x10, &regs, &regs);
 isega=(regs.x.cx!=0xFFFF);
 regs.h.ah=0x0F;
 int86(0x10, &regs, &regs);
 mda=regs.h.al;
 columns=regs.h.ah;
 if (mda==7)
 {
  isega=1;
  mda=1;
 }
 else
 {
  if ((mda<2)||(mda>3))
  {
   regs.x.ax=0x0003-(!mda);
   int86(0x10, &regs, &regs);
   
   columns=80;
   lines=25;
  }
  else if (isega)
  {
   lines=*(uint8_t far *)(MK_FP(0x0040, 0x0084))+1;
  }
  mda=0;
 }
 
 regs.h.ah=0x03;
 regs.h.bh=0x00;
 int86(0x10, &regs, &regs);
 csrstat=regs.x.cx&0x1F1F;
 
 for (y=0; y<lines; y++)
  for (x=0; x<columns; x++)
   putchattr(178, 0x70, x, y);
}

void deinitty (void)
{
 uint8_t far *a;
 uint16_t w, t;
 
 /* Restore the cursor */
 regs.h.ah=0x01;
 regs.x.cx=csrstat;
 int86(0x10, &regs, &regs);
 
 /* Clear the screen */
 regs.x.ax=0x0600;
 regs.h.bh=0x07;
 regs.x.cx=0x0000;
 regs.h.dh=lines-1;
 regs.h.dl=columns-1;
 int86(0x10, &regs, &regs);
 
 /* Jump cursor to top left */
 regs.h.ah=0x02;
 regs.h.bh=0x00;
 regs.x.dx=0x0000;
 int86(0x10, &regs, &regs);
}
